<?php

namespace Database\Factories;

use App\Enums\PatrollingLocationType;
use App\Models\PatrollingLocation;
use App\Models\Wing;
use Illuminate\Database\Eloquent\Factories\Factory;

class PatrollingLocationFactory extends Factory
{
    protected $model = PatrollingLocation::class;

    public function definition(): array
    {
        $type = $this->faker->randomElement(PatrollingLocationType::cases());
        
        return [
            'name' => $this->getLocationName($type),
            'type' => $type,
            'location_code' => $this->faker->unique()->regexify('[A-Z]{2}[0-9]{3}'),
            'description' => $this->faker->optional()->sentence(),
            'is_active' => $this->faker->boolean(95), // 95% chance of being active
            'wing_id' => $this->faker->optional()->passthrough(Wing::factory()),
            'coordinates' => $this->faker->optional()->latitude . ',' . $this->faker->optional()->longitude,
        ];
    }

    private function getLocationName(PatrollingLocationType $type): string
    {
        $names = match ($type) {
            PatrollingLocationType::GATE => ['Main Gate', 'Side Gate', 'Emergency Gate', 'Service Gate'],
            PatrollingLocationType::GARDEN => ['Central Garden', 'Rose Garden', 'Kids Play Garden', 'Meditation Garden'],
            PatrollingLocationType::PARKING => ['Basement Parking', 'Ground Floor Parking', 'Visitor Parking', 'Two Wheeler Parking'],
            PatrollingLocationType::COMMON_AREA => ['Reception', 'Lobby', 'Common Hall', 'Reading Room'],
            PatrollingLocationType::BUILDING => ['A Wing Entrance', 'B Wing Entrance', 'C Wing Entrance', 'Service Building'],
            PatrollingLocationType::PLAYGROUND => ['Cricket Ground', 'Basketball Court', 'Tennis Court', 'Kids Playground'],
            PatrollingLocationType::SWIMMING_POOL => ['Main Pool', 'Kids Pool', 'Pool Deck', 'Pool Equipment Room'],
            PatrollingLocationType::GYM => ['Main Gym', 'Cardio Room', 'Weight Room', 'Yoga Room'],
            PatrollingLocationType::CLUB_HOUSE => ['Main Hall', 'Party Hall', 'Meeting Room', 'Library'],
            PatrollingLocationType::OTHER => ['Security Cabin', 'Pump Room', 'Generator Room', 'Garbage Area'],
        };
        
        return $this->faker->randomElement($names);
    }

    public function active(): static
    {
        return $this->state(fn (array $attributes) => [
            'is_active' => true,
        ]);
    }

    public function inactive(): static
    {
        return $this->state(fn (array $attributes) => [
            'is_active' => false,
        ]);
    }

    public function gate(): static
    {
        return $this->state(fn (array $attributes) => [
            'type' => PatrollingLocationType::GATE,
            'name' => $this->getLocationName(PatrollingLocationType::GATE),
        ]);
    }

    public function garden(): static
    {
        return $this->state(fn (array $attributes) => [
            'type' => PatrollingLocationType::GARDEN,
            'name' => $this->getLocationName(PatrollingLocationType::GARDEN),
        ]);
    }

    public function parking(): static
    {
        return $this->state(fn (array $attributes) => [
            'type' => PatrollingLocationType::PARKING,
            'name' => $this->getLocationName(PatrollingLocationType::PARKING),
        ]);
    }
}