<?php

namespace App\Http\Requests\DailyPatrolSummary;

use App\Http\Requests\BaseRequest;
use App\Enums\DailyPatrolSummaryStatus;
use Illuminate\Validation\Rule;

class UpdateDailyPatrolSummaryRequest extends BaseRequest
{
    public function authorize(): bool
    {
        return true;
    }

    public function rules(): array
    {
        return [
            'guard_id' => ['sometimes', 'exists:guards,id'],
            'patrol_round_id' => ['sometimes', 'exists:patrol_rounds,id'],
            'patrol_date' => ['sometimes', 'date'],
            'total_locations' => ['sometimes', 'integer', 'min:0'],
            'completed_locations' => ['sometimes', 'integer', 'min:0'],
            'missed_locations' => ['sometimes', 'integer', 'min:0'],
            'expected_start_time' => ['sometimes', 'date_format:H:i'],
            'expected_end_time' => ['sometimes', 'date_format:H:i'],
            'actual_start_time' => ['sometimes', 'nullable', 'date'],
            'actual_end_time' => ['sometimes', 'nullable', 'date'],
            'status' => ['sometimes', Rule::enum(DailyPatrolSummaryStatus::class)],
            'notes' => ['sometimes', 'nullable', 'string', 'max:1000'],
        ];
    }

    public function messages(): array
    {
        return [
            'guard_id.exists' => 'Selected guard does not exist',
            'patrol_round_id.exists' => 'Selected patrol round does not exist',
            'patrol_date.date' => 'Patrol date must be a valid date',
            'total_locations.min' => 'Total locations must be at least 0',
            'expected_start_time.date_format' => 'Expected start time must be in HH:MM format',
            'expected_end_time.date_format' => 'Expected end time must be in HH:MM format',
            'actual_start_time.date' => 'Actual start time must be a valid date',
            'actual_end_time.date' => 'Actual end time must be a valid date',
        ];
    }

    protected function prepareForValidation(): void
    {
        // Calculate missed locations if total and completed are being updated
        if ($this->has('total_locations') && $this->has('completed_locations')) {
            $this->merge([
                'missed_locations' => max(0, $this->total_locations - $this->completed_locations)
            ]);
        }

        // Validate time relationships
        if ($this->has('expected_start_time') && $this->has('expected_end_time')) {
            $startTime = $this->expected_start_time;
            $endTime = $this->expected_end_time;
            
            if ($startTime >= $endTime) {
                $this->merge(['expected_end_time' => null]); // This will trigger validation error
            }
        }

        if ($this->has('actual_start_time') && $this->has('actual_end_time')) {
            $actualStart = $this->actual_start_time;
            $actualEnd = $this->actual_end_time;
            
            if ($actualStart && $actualEnd && $actualStart >= $actualEnd) {
                $this->merge(['actual_end_time' => null]); // This will trigger validation error
            }
        }
    }
}