<?php

namespace App\Models;

use App\Enums\DailyPatrolSummaryStatus;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class DailyPatrolSummary extends Model
{
    use HasFactory;

    protected $fillable = [
        'guard_id',
        'patrol_round_id',
        'patrol_date',
        'total_locations',
        'completed_locations',
        'missed_locations',
        'expected_start_time',
        'expected_end_time',
        'actual_start_time',
        'actual_end_time',
        'status',
        'notes',
    ];

    protected function casts(): array
    {
        return [
            'patrol_date' => 'date',
            'expected_start_time' => 'datetime:H:i',
            'expected_end_time' => 'datetime:H:i',
            'actual_start_time' => 'datetime',
            'actual_end_time' => 'datetime',
            'status' => DailyPatrolSummaryStatus::class,
        ];
    }

    public function guardUser(): BelongsTo
    {
        return $this->belongsTo(User::class, 'guard_id');
    }

    public function patrolRound(): BelongsTo
    {
        return $this->belongsTo(PatrolRound::class);
    }

    public function patrollingLogs(): HasMany
    {
        return $this->hasMany(PatrollingLog::class);
    }

    public function getActualDurationAttribute(): ?string
    {
        if (!$this->actual_start_time || !$this->actual_end_time) {
            return null;
        }

        $diff = $this->actual_start_time->diff($this->actual_end_time);
        return $diff->format('%H:%I');
    }

    public function getExpectedDurationAttribute(): ?string
    {
        if (!$this->expected_start_time || !$this->expected_end_time) {
            return null;
        }

        $diff = $this->expected_start_time->diff($this->expected_end_time);
        return $diff->format('%H:%I');
    }

    public function getIsOnTimeAttribute(): bool
    {
        if (!$this->actual_start_time || !$this->expected_start_time) {
            return false;
        }

        return $this->actual_start_time <= $this->expected_start_time->addMinutes(15); // 15 min tolerance
    }

    public function getIsDelayedAttribute(): bool
    {
        return !$this->is_on_time;
    }

    public function getCompletionPercentageAttribute(): float
    {
        if ($this->total_locations === 0) {
            return 0;
        }

        return round(($this->completed_locations / $this->total_locations) * 100, 2);
    }

    // Query scopes for filtering
    public function scopeToday($query)
    {
        return $query->where('patrol_date', now()->format('Y-m-d'));
    }

    public function scopeThisWeek($query)
    {
        return $query->whereBetween('patrol_date', [
            now()->startOfWeek()->format('Y-m-d'),
            now()->endOfWeek()->format('Y-m-d')
        ]);
    }

    public function scopeThisMonth($query)
    {
        return $query->whereBetween('patrol_date', [
            now()->startOfMonth()->format('Y-m-d'),
            now()->endOfMonth()->format('Y-m-d')
        ]);
    }
}
