<?php

namespace App\Traits;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

trait CanManageFile
{
    protected ?string $resourceName = null;

    protected array $fileFieldNames = [];

    protected array $fileFolderPaths = [];

    /**
     * Store multiple files.
     */
    protected function storeFiles(Request $request, Model $model, string $disk = 'public'): void {
        foreach ($this->fileFieldNames as $i => $fieldName) {
            if ($request->hasFile($fieldName)) {
                $folderPath = $this->buildFolderPath($this->fileFolderPaths[$i] ?? $fieldName);
                $model->{$fieldName} = $this->storeFile($request->file($fieldName), $folderPath, $disk);
            }
        }
    }

    /**
     * Build the folder path, optionally prepending the resource name if it is set.
     */
    protected function buildFolderPath(string $folderPath): string {
        return $this->resourceName ? $this->resourceName.'/'.$folderPath : $folderPath;
    }

    /**
     * Store a single file and return its full URL.
     */
    public function storeFile($file, string $folderPath, string $disk = 'public'): string {
        // Generate a unique name for the file
        $fileName = time().$file->hashName();

        // Store the file in the desired folder on the specified disk
        $file->storeAs($folderPath, $fileName, $disk);

        // Generate and return the full URL
        return Storage::disk($disk)->url($folderPath.'/'.$fileName);
    }

    /**
     * Update multiple files.
     */
    protected function updateFiles(Request $request, Model $model, string $disk = 'public'): void {
        foreach ($this->fileFieldNames as $i => $fieldName) {
            if ($request->hasFile($fieldName)) {
                // Delete old file if it exists
                $folderPath = $this->buildFolderPath($this->fileFolderPaths[$i] ?? $fieldName);
                $this->deleteFile($model[$fieldName], $folderPath, $disk);
                // Upload new file and store its full URL
                $model->{$fieldName} = $this->storeFile($request->file($fieldName), $folderPath, $disk);
            }
        }
    }

    /**
     * Delete a single file.
     */
    public function deleteFile(?string $fileName, string $folderPath, string $disk = 'public'): void {
        if ($fileName) {
            // Check if $fileName is a full URL; extract the relative path if necessary
            $relativePath = $this->extractRelativePath($fileName, $folderPath, $disk);
            $filePath = $folderPath.'/'.$relativePath;
            // Delete the file if it exists on the disk
            if (Storage::disk($disk)->exists($filePath)) {
                Storage::disk($disk)->delete($filePath);
            }
        }
    }

    /**
     * Extract the relative path from a URL or use the filename directly.
     */
    protected function extractRelativePath(?string $fileName, string $folderPath, string $disk): string {
        if (!$fileName) {
            return '';
        }

        // If the filename is a URL, extract the relative path
        $urlPrefix = Storage::disk($disk)->url($folderPath.'/');
        if (str_starts_with($fileName, $urlPrefix)) {
            return basename($fileName);
        }

        // If it's already a relative path (for backward compatibility), return it as is
        return $fileName;
    }

    /**
     * Delete multiple files.
     */
    protected function deleteFiles(Model $model, string $disk = 'public'): void {
        foreach ($this->fileFieldNames as $i => $fieldName) {
            $folderPath = $this->buildFolderPath($this->fileFolderPaths[$i] ?? $fieldName);
            $this->deleteFile($model[$fieldName], $folderPath, $disk);
        }
    }
}
