# Resident Patrol API Documentation

## Overview

The Resident Patrol API is a comprehensive Laravel-based REST API designed for residential security management. It provides functionality for user authentication, visitor management, vehicle tracking, security patrolling, and residential administration.

**Base URL:** `{your-domain}/api`  
**Authentication:** Laravel Sanctum (Bearer Token)

## Table of Contents

1. [Authentication](#authentication)
2. [User Management](#user-management)
3. [Wing Management](#wing-management)
4. [Owner Management](#owner-management)
5. [Flat Management](#flat-management)
6. [Guard Management](#guard-management)
7. [Vehicle Management](#vehicle-management)
8. [Vehicle Log Management](#vehicle-log-management)
9. [Patrolling Location Management](#patrolling-location-management)
10. [Patrolling Log Management](#patrolling-log-management)
11. [Visitor Management](#visitor-management)
12. [Data Models](#data-models)
13. [Enums & Constants](#enums--constants)
14. [Error Handling](#error-handling)

## Authentication

### Register User
**POST** `/register`

Create a new user account.

**Request Body:**
```json
{
  "owner_id": 1,              // optional, integer
  "flat_id": 1,               // optional, integer
  "name": "John Doe",         // required, string
  "email": "john@example.com", // required, string, unique
  "phone": "+1234567890",     // required, string
  "password": "password123",   // required, string
  "password_confirmation": "password123", // required, string
  "role": "user"              // required, enum: admin|owner|user|guard
}
```

**Response:**
```json
{
  "success": true,
  "message": "User registered successfully"
}
```

### Login
**POST** `/login`

Authenticate user and receive access token.

**Request Body:**
```json
{
  "email": "john@example.com", // optional (either email or phone required)
  "phone": "+1234567890",      // optional (either email or phone required)
  "password": "password123"    // required, string
}
```

**Response:**
```json
{
  "success": true,
  "message": "Login successful",
  "data": {
    "user": {
      "id": 1,
      "name": "John Doe",
      "email": "john@example.com",
      "role": "user"
    },
    "token": "1|abcdef123456..."
  }
}
```

### Forgot Password
**POST** `/forgot-password`

Request password reset email.

**Request Body:**
```json
{
  "email": "john@example.com" // required, string
}
```

### Update Password
**POST** `/update-password`

Update password using reset token.

**Request Body:**
```json
{
  "token": "reset-token-here",      // required, string
  "password": "newpassword123",     // required, string
  "confirmation_password": "newpassword123" // required, string
}
```

### Get Profile
**GET** `/users/profile`

Get current authenticated user's profile.

**Headers:**
```
Authorization: Bearer {token}
```

**Response:**
```json
{
  "success": true,
  "data": {
    "id": 1,
    "name": "John Doe",
    "email": "john@example.com",
    "phone": "+1234567890",
    "role": "user",
    "created_at": "2025-07-11T10:30:00Z"
  }
}
```

### Logout
**DELETE** `/logout`

Invalidate the current access token.

**Headers:**
```
Authorization: Bearer {token}
```

## User Management

All user management endpoints require authentication.

### List Users
**GET** `/users`

**Query Parameters:**
- `owner_id` - Filter by owner ID
- `flat_id` - Filter by flat ID
- `name` - Filter by name
- `email` - Filter by email
- `phone` - Filter by phone
- `role` - Filter by role (admin|owner|user|guard)
- `gender` - Filter by gender (male|female|other)
- `include` - Include relationships (owner,flat)
- `perPage` - Number of items per page (default: 15)

**Response:**
```json
{
  "success": true,
  "data": [
    {
      "id": 1,
      "name": "John Doe",
      "email": "john@example.com",
      "phone": "+1234567890",
      "role": "user",
      "gender": "male",
      "relation": "self",
      "photo": "/storage/photos/user1.jpg",
      "owner_id": 1,
      "flat_id": 1,
      "created_at": "2025-07-11T10:30:00Z",
      "updated_at": "2025-07-11T10:30:00Z"
    }
  ],
  "meta": {
    "current_page": 1,
    "total": 50,
    "per_page": 15
  }
}
```

### Create User
**POST** `/users`

**Request Body (multipart/form-data):**
```json
{
  "owner_id": 1,              // optional, integer
  "flat_id": 1,               // optional, integer
  "name": "John Doe",         // required, string
  "email": "john@example.com", // required, string, unique
  "phone": "+1234567890",     // required, string
  "password": "password123",   // required, string
  "password_confirmation": "password123", // required, string
  "role": "user",             // required, enum
  "gender": "male",           // required, enum
  "relation": "self",         // optional, string
  "photo": "file"             // optional, file (max 5MB)
}
```

### Get User
**GET** `/users/{id}`

### Update User
**PUT** `/users/{id}`

Same fields as create, but all optional except validation rules still apply for provided fields.

### Delete User
**DELETE** `/users/{id}`

## Wing Management

### List Wings
**GET** `/wings`

**Query Parameters:**
- `name` - Filter by wing name
- `perPage` - Number of items per page (default: 15)

### Create Wing
**POST** `/wings`

**Request Body:**
```json
{
  "name": "Wing A",                    // required, string
  "no_floors": 10,                     // required, integer
  "no_flats": 40,                      // required, integer
  "no_lifts": 2,                       // required, integer
  "fire_extinguisher_date": "2025-12-31" // required, date (Y-m-d)
}
```

### Get Wing
**GET** `/wings/{id}`

### Update Wing
**PUT** `/wings/{id}`

### Delete Wing
**DELETE** `/wings/{id}`

## Owner Management

### List Owners
**GET** `/owners`

**Query Parameters:**
- `name` - Filter by owner name
- `email` - Filter by email
- `phone` - Filter by phone
- `include` - Include relationships (society)
- `perPage` - Number of items per page (default: 15)

### Create Owner
**POST** `/owners`

**Request Body (multipart/form-data):**
```json
{
  "name": "John Smith",           // required, string
  "email": "john@example.com",    // required, string, unique
  "phone": "+1234567890",         // required, string
  "aadhaar_no": "123456789012",   // required, string, unique
  "pan_no": "ABCDE1234F",         // required, string, unique
  "aadhaar_card": "file",         // optional, file (max 5MB)
  "pan_card": "file",             // optional, file (max 5MB)
  "photo": "file"                 // optional, file (max 5MB)
}
```

### Get Owner
**GET** `/owners/{id}`

### Update Owner
**PUT** `/owners/{id}`

### Delete Owner
**DELETE** `/owners/{id}`

## Flat Management

### List Flats
**GET** `/flats`

**Query Parameters:**
- `wing_id` - Filter by wing ID
- `owner_id` - Filter by owner ID
- `flat_no` - Filter by flat number
- `status` - Filter by status (active|in_active)
- `include` - Include relationships (wing,owner)
- `perPage` - Number of items per page (default: 15)

**Note:** USER and OWNER roles can only see their own flats.

### Create Flat
**POST** `/flats`

**Request Body (multipart/form-data):**
```json
{
  "wing_id": 1,                // required, integer
  "owner_id": 1,               // optional, integer
  "flat_no": "A-101",          // required, string
  "status": "active",          // required, enum (active|in_active)
  "flat_sqrft": 1200.50,       // required, numeric
  "agreement": "file",         // optional, file (max 5MB)
  "other_docs": "file"         // optional, file (max 5MB)
}
```

### Get Flat
**GET** `/flats/{id}`

### Update Flat
**PUT** `/flats/{id}`

### Delete Flat
**DELETE** `/flats/{id}`

## Guard Management

### List Guards
**GET** `/guards`

**Query Parameters:**
- `wing_id` - Filter by wing ID
- `name` - Filter by name
- `email` - Filter by email
- `role` - Filter by role (watchman|other)
- `phone` - Filter by phone
- `gender` - Filter by gender (male|female|other)
- `include` - Include relationships (wing)
- `perPage` - Number of items per page (default: 15)

### Create Guard
**POST** `/guards`

**Request Body (multipart/form-data):**
```json
{
  "wing_id": 1,                // optional, integer
  "name": "Security Guard",    // required, string
  "email": "guard@example.com", // required, string, unique
  "phone": "+1234567890",      // required, string
  "role": "watchman",          // required, enum (watchman|other)
  "gender": "male",            // required, enum (male|female|other)
  "photo": "file",             // optional, file (max 5MB)
  "agreement": "file"          // optional, file (max 5MB)
}
```

### Get Guard
**GET** `/guards/{id}`

### Update Guard
**PUT** `/guards/{id}`

### Delete Guard
**DELETE** `/guards/{id}`

## Vehicle Management

### List Vehicles
**GET** `/vehicles`

**Query Parameters:**
- `owner_id` - Filter by owner ID
- `flat_id` - Filter by flat ID
- `vehicle_type` - Filter by type (two_wheeler|four_wheeler|other)
- `make` - Filter by make
- `model` - Filter by model
- `registration_no` - Filter by registration number
- `is_active` - Filter by active status (true|false)
- `include` - Include relationships (owner,flat)
- `perPage` - Number of items per page (default: 15)

### Create Vehicle
**POST** `/vehicles`

**Request Body (multipart/form-data):**
```json
{
  "owner_id": 1,                    // required, integer
  "flat_id": 1,                     // required, integer
  "vehicle_type": "four_wheeler",   // required, enum
  "make": "Honda",                  // required, string
  "model": "Civic",                 // required, string
  "color": "Blue",                  // required, string
  "registration_no": "MH12AB1234",  // required, string, unique
  "photo": "file",                  // optional, file (max 5MB)
  "insurance_doc": "file",          // optional, file (max 5MB)
  "is_active": true                 // optional, boolean (default: true)
}
```

**Response:** Vehicle with auto-generated `unique_code` (format: VEH + 8 random characters)

### Get Vehicle
**GET** `/vehicles/{id}`

### Update Vehicle
**PUT** `/vehicles/{id}`

### Delete Vehicle
**DELETE** `/vehicles/{id}`

## Vehicle Log Management

### List Vehicle Logs
**GET** `/vehicle-logs`

**Query Parameters:**
- `vehicle_id` - Filter by vehicle ID
- `guard_id` - Filter by guard ID
- `entry_type` - Filter by type (entry|exit)
- `entry_time` - Filter by entry time
- `exit_time` - Filter by exit time
- `include` - Include relationships (vehicle,guardUser)
- `perPage` - Number of items per page (default: 15)

### Create Vehicle Log
**POST** `/vehicle-logs`

**Request Body:**
```json
{
  "vehicle_id": 1,               // optional, integer
  "guard_id": 1,                 // required, integer
  "entry_type": "entry",         // required, enum (entry|exit)
  "entry_time": "2025-07-11T10:30:00Z", // required, datetime
  "exit_time": "2025-07-11T18:30:00Z",  // optional, datetime (must be after entry_time)
  "purpose": "Office visit",     // optional, string
  "visitor_details": {           // optional, object
    "name": "John Visitor",
    "phone": "+1234567890"
  },
  "outsider_vehicle_details": {  // optional, object
    "vehicle_type": "four_wheeler",
    "make": "Toyota",
    "model": "Camry",
    "color": "White",
    "registration_no": "MH12CD5678"
  }
}
```

### Scan Vehicle
**POST** `/vehicle-logs/scan`

**Access:** GUARD role only

**Request Body:**
```json
{
  "unique_code": "VEH12345678",      // required, string
  "entry_type": "entry",             // required, enum (entry|exit)
  "purpose": "Resident return",      // optional, string
  "outsider_vehicle_details": {}     // optional, object (for non-registered vehicles)
}
```

### Get Today's Vehicle Logs
**GET** `/vehicle-logs/today`

**Query Parameters:**
- `vehicle_id` - Filter by vehicle ID
- `guard_id` - Filter by guard ID
- `entry_type` - Filter by type (entry|exit)

### Get Vehicle Log
**GET** `/vehicle-logs/{id}`

### Update Vehicle Log
**PUT** `/vehicle-logs/{id}`

### Delete Vehicle Log
**DELETE** `/vehicle-logs/{id}`

## Patrolling Location Management

### List Patrolling Locations
**GET** `/patrolling-locations`

**Query Parameters:**
- `name` - Filter by name
- `type` - Filter by location type
- `location_code` - Filter by location code
- `is_active` - Filter by active status (true|false)
- `wing_id` - Filter by wing ID
- `include` - Include relationships (wing)
- `perPage` - Number of items per page (default: 15)

### Create Patrolling Location
**POST** `/patrolling-locations`

**Request Body:**
```json
{
  "name": "Main Gate",              // required, string
  "type": "gate",                   // required, enum (see location types)
  "description": "Primary entrance", // optional, string
  "is_active": true,                // optional, boolean (default: true)
  "wing_id": 1,                     // optional, integer
  "coordinates": "18.5204,73.8567"  // optional, string
}
```

**Response:** Location with auto-generated `location_code` (format: LOC + 8 random characters)

### Get Patrolling Location
**GET** `/patrolling-locations/{id}`

### Update Patrolling Location
**PUT** `/patrolling-locations/{id}`

### Delete Patrolling Location
**DELETE** `/patrolling-locations/{id}`

## Patrolling Log Management

### List Patrolling Logs
**GET** `/patrolling-logs`

**Query Parameters:**
- `location_id` - Filter by location ID
- `guard_id` - Filter by guard ID
- `patrol_date` - Filter by patrol date
- `status` - Filter by status (checked|issue_found|maintenance_required|not_accessible)
- `include` - Include relationships (location,guardUser)
- `perPage` - Number of items per page (default: 15)

### Create Patrolling Log
**POST** `/patrolling-logs`

**Request Body:**
```json
{
  "location_id": 1,                    // required, integer
  "guard_id": 1,                       // required, integer
  "patrol_date": "2025-07-11",         // required, date
  "patrol_time": "2025-07-11 14:30",   // required, datetime (Y-m-d H:i)
  "status": "checked",                 // required, enum
  "notes": "All clear",                // optional, string
  "issues_found": "Minor crack in wall" // optional, string
}
```

### Scan Location
**POST** `/patrolling-logs/scan`

**Request Body:**
```json
{
  "location_code": "LOC12345678",      // required, string
  "status": "checked",                 // required, enum
  "notes": "Patrol completed",         // optional, string
  "issues_found": "No issues"          // optional, string
}
```

### Get Today's Patrolling Logs
**GET** `/patrolling-logs/today`

**Query Parameters:**
- `location_id` - Filter by location ID
- `guard_id` - Filter by guard ID
- `status` - Filter by status

### Get Missed Locations
**GET** `/patrolling-logs/missed-locations`

Returns active locations not patrolled today.

### Start Daily Round
**POST** `/patrolling-logs/start-daily-round`

Creates or updates DailyPatrolSummary with STARTED status.

### Complete Daily Round
**POST** `/patrolling-logs/complete-daily-round`

**Request Body:**
```json
{
  "notes": "Daily round completed successfully" // optional, string
}
```

Updates summary with completion statistics and COMPLETED status.

### Get Patrolling Log
**GET** `/patrolling-logs/{id}`

### Update Patrolling Log
**PUT** `/patrolling-logs/{id}`

### Delete Patrolling Log
**DELETE** `/patrolling-logs/{id}`

## Visitor Management

### List Visitors
**GET** `/visitors`

**Query Parameters:**
- `name` - Filter by visitor name
- `phone` - Filter by phone number
- `status` - Filter by status (pending|approved|rejected|checked_in|checked_out|expired)
- `visitor_type` - Filter by type (guest|service_provider|delivery|contractor|family|friend|other)
- `wing_id` - Filter by wing ID
- `flat_id` - Filter by flat ID
- `approved_by` - Filter by approver user ID
- `include` - Include relationships (wing,flat,approvedBy,checkedInBy,checkedOutBy)
- `sort` - Sort by fields (created_at,entry_time,exit_time,approved_at)
- `perPage` - Number of items per page (default: 15)

### Create Visitor Request
**POST** `/visitors`

**Request Body (multipart/form-data):**
```json
{
  "wing_id": 1,                        // required, integer
  "flat_id": 1,                        // required, integer
  "name": "Jane Visitor",              // required, string
  "phone": "+1234567890",              // required, string
  "email": "jane@example.com",         // optional, string
  "visitor_type": "guest",             // required, enum
  "purpose": "Social visit",           // required, string (max 500 chars)
  "photo": "file",                     // optional, image file (max 2MB)
  "entry_time": "2025-07-11T14:00:00Z", // required, datetime (future)
  "exit_time": "2025-07-11T18:00:00Z",  // optional, datetime (after entry_time)
  "additional_info": {},               // optional, object
  "is_recurring": false                // optional, boolean
}
```

### Get Visitor
**GET** `/visitors/{id}`

### Update Visitor
**PUT** `/visitors/{id}`

**Note:** Only allowed if visitor status is PENDING.

### Delete Visitor
**DELETE** `/visitors/{id}`

**Note:** Not allowed if visitor is checked in.

### Approve Visitor
**POST** `/visitors/{id}/approve`

**Access:** Flat owner only

**Request Body:**
```json
{
  "approval_notes": "Approved for social visit" // optional, string (max 1000 chars)
}
```

**Status Change:** PENDING → APPROVED

### Reject Visitor
**POST** `/visitors/{id}/reject`

**Access:** Flat owner only

**Request Body:**
```json
{
  "rejection_reason": "Not available at requested time" // required, string (max 1000 chars)
}
```

**Status Change:** PENDING → REJECTED

### Check In Visitor
**POST** `/visitors/{id}/check-in`

**Access:** Any authenticated user

**Status Change:** APPROVED → CHECKED_IN

### Check Out Visitor
**POST** `/visitors/{id}/check-out`

**Access:** Any authenticated user

**Status Change:** CHECKED_IN → CHECKED_OUT

### Get Pending Visitors
**GET** `/visitors/pending`

Returns visitors with PENDING status for flats owned by current user.

### Get Today's Visitors
**GET** `/visitors/today`

Returns visitors with entry_time for today.

### Get My Visitors
**GET** `/visitors/my-visitors`

Returns all visitors for flats owned by current user.

## Data Models

### User Model
```json
{
  "id": "integer",
  "owner_id": "integer|null",
  "flat_id": "integer|null", 
  "name": "string",
  "email": "string",
  "phone": "string",
  "role": "enum(admin|owner|user|guard)",
  "gender": "enum(male|female|other)",
  "relation": "string|null",
  "photo": "string|null",
  "email_verified_at": "timestamp|null",
  "created_at": "timestamp",
  "updated_at": "timestamp"
}
```

### Wing Model
```json
{
  "id": "integer",
  "name": "string",
  "no_floors": "integer",
  "no_flats": "integer", 
  "no_lifts": "integer",
  "fire_extinguisher_date": "date",
  "created_at": "timestamp",
  "updated_at": "timestamp"
}
```

### Flat Model
```json
{
  "id": "integer",
  "wing_id": "integer",
  "owner_id": "integer|null",
  "flat_no": "string",
  "status": "enum(active|in_active)",
  "flat_sqrft": "decimal",
  "agreement": "string|null",
  "other_docs": "string|null",
  "created_at": "timestamp",
  "updated_at": "timestamp"
}
```

### Owner Model
```json
{
  "id": "integer",
  "name": "string",
  "email": "string",
  "phone": "string",
  "aadhaar_no": "string",
  "pan_no": "string",
  "created_at": "timestamp",
  "updated_at": "timestamp"
}
```

### Guard Model
```json
{
  "id": "integer",
  "wing_id": "integer|null",
  "name": "string",
  "email": "string",
  "role": "enum(watchman|other)",
  "phone": "string",
  "gender": "enum(male|female|other)",
  "photo": "string|null",
  "agreement": "string|null",
  "created_at": "timestamp",
  "updated_at": "timestamp"
}
```

### Vehicle Model
```json
{
  "id": "integer",
  "owner_id": "integer",
  "flat_id": "integer",
  "vehicle_type": "enum(two_wheeler|four_wheeler|other)",
  "make": "string",
  "model": "string",
  "color": "string",
  "registration_no": "string",
  "unique_code": "string",
  "photo": "string|null",
  "insurance_doc": "string|null",
  "is_active": "boolean",
  "created_at": "timestamp",
  "updated_at": "timestamp"
}
```

### VehicleLog Model
```json
{
  "id": "integer",
  "vehicle_id": "integer|null",
  "guard_id": "integer",
  "entry_type": "enum(entry|exit)",
  "entry_time": "datetime",
  "exit_time": "datetime|null",
  "purpose": "string|null",
  "visitor_details": "json|null",
  "outsider_vehicle_details": "json|null",
  "created_at": "timestamp",
  "updated_at": "timestamp"
}
```

### PatrollingLocation Model
```json
{
  "id": "integer",
  "name": "string",
  "type": "enum(gate|garden|parking|common_area|building|playground|swimming_pool|gym|club_house|other)",
  "location_code": "string",
  "description": "text|null",
  "is_active": "boolean",
  "wing_id": "integer|null",
  "coordinates": "string|null",
  "created_at": "timestamp",
  "updated_at": "timestamp"
}
```

### PatrollingLog Model
```json
{
  "id": "integer",
  "location_id": "integer",
  "guard_id": "integer",
  "patrol_date": "date",
  "patrol_time": "datetime",
  "status": "enum(checked|issue_found|maintenance_required|not_accessible)",
  "notes": "text|null",
  "issues_found": "text|null",
  "created_at": "timestamp",
  "updated_at": "timestamp"
}
```

### DailyPatrolSummary Model
```json
{
  "id": "integer",
  "guard_id": "integer",
  "patrol_date": "date",
  "total_locations": "integer",
  "completed_locations": "integer",
  "missed_locations": "integer",
  "start_time": "datetime|null",
  "end_time": "datetime|null",
  "status": "enum(not_started|started|completed)",
  "notes": "text|null",
  "created_at": "timestamp",
  "updated_at": "timestamp"
}
```

### Visitor Model
```json
{
  "id": "integer",
  "name": "string",
  "phone": "string",
  "email": "string|null",
  "visitor_type": "enum(guest|service_provider|delivery|contractor|family|friend|other)",
  "wing_id": "integer",
  "flat_id": "integer",
  "purpose": "string",
  "photo": "text|null",
  "status": "enum(pending|approved|rejected|checked_in|checked_out|expired)",
  "entry_time": "datetime|null",
  "exit_time": "datetime|null",
  "approved_by": "integer|null",
  "approved_at": "datetime|null",
  "approval_notes": "text|null",
  "rejection_reason": "text|null",
  "checked_in_by": "integer|null",
  "checked_out_by": "integer|null",
  "additional_info": "json|null",
  "is_recurring": "boolean",
  "created_at": "timestamp",
  "updated_at": "timestamp"
}
```

## Enums & Constants

### User Roles
- `admin` - Administrator with full access
- `owner` - Property owner with management rights
- `user` - Regular resident with limited access
- `guard` - Security guard with patrol and entry/exit management

### Gender Options
- `male`
- `female` 
- `other`

### Flat Status
- `active` - Active flat
- `in_active` - Inactive flat

### Guard Roles
- `watchman` - Regular security guard
- `other` - Other guard roles

### Vehicle Types
- `two_wheeler` - Motorcycles, scooters
- `four_wheeler` - Cars, SUVs
- `other` - Other vehicle types

### Vehicle Log Types
- `entry` - Vehicle entering premises
- `exit` - Vehicle leaving premises

### Patrolling Location Types
- `gate` - Entry/exit gates
- `garden` - Garden areas
- `parking` - Parking areas
- `common_area` - Common areas
- `building` - Building areas
- `playground` - Playground areas
- `swimming_pool` - Swimming pool area
- `gym` - Gym facility
- `club_house` - Club house
- `other` - Other location types

### Patrolling Status
- `checked` - Location checked successfully
- `issue_found` - Issues found during patrol
- `maintenance_required` - Maintenance needed
- `not_accessible` - Location not accessible

### Daily Patrol Summary Status
- `not_started` - Patrol not started
- `started` - Patrol in progress
- `completed` - Patrol completed

### Visitor Types
- `guest` - Personal guest
- `service_provider` - Service provider
- `delivery` - Delivery person
- `contractor` - Contractor/worker
- `family` - Family member
- `friend` - Friend
- `other` - Other visitor types

### Visitor Status
- `pending` - Awaiting approval
- `approved` - Approved by flat owner
- `rejected` - Rejected by flat owner
- `checked_in` - Visitor has checked in
- `checked_out` - Visitor has checked out
- `expired` - Visitor entry time has passed

## Error Handling

### Standard HTTP Status Codes

- **200 OK** - Successful request
- **201 Created** - Resource created successfully
- **204 No Content** - Successful request with no content (e.g., delete)
- **400 Bad Request** - Invalid request format
- **401 Unauthorized** - Authentication required
- **403 Forbidden** - Insufficient permissions
- **404 Not Found** - Resource not found
- **422 Unprocessable Entity** - Validation errors
- **500 Internal Server Error** - Server error

### Error Response Format

```json
{
  "success": false,
  "message": "Error description",
  "errors": {
    "field_name": ["Validation error message"]
  }
}
```

### Common Error Scenarios

1. **Authentication Errors**
   - Missing or invalid token
   - Expired token

2. **Validation Errors**
   - Required fields missing
   - Invalid data formats
   - Constraint violations (unique fields, etc.)

3. **Authorization Errors**
   - Insufficient role permissions
   - Attempting to access another user's data

4. **Business Logic Errors**
   - Visitor already checked in/out
   - Attempting to approve own visitor request
   - Updating visitor after check-in

### File Upload Errors

- File size exceeds 5MB (or 2MB for visitor photos)
- Unsupported file format
- Missing file when required

## Rate Limiting

The API implements standard Laravel rate limiting:
- **Guest requests:** 60 requests per minute
- **Authenticated requests:** 60 requests per minute per user

## Response Pagination

Paginated endpoints return data in the following format:

```json
{
  "success": true,
  "data": [...],
  "meta": {
    "current_page": 1,
    "from": 1,
    "last_page": 10,
    "path": "http://example.com/api/endpoint",
    "per_page": 15,
    "to": 15,
    "total": 150
  },
  "links": {
    "first": "http://example.com/api/endpoint?page=1",
    "last": "http://example.com/api/endpoint?page=10",
    "prev": null,
    "next": "http://example.com/api/endpoint?page=2"
  }
}
```

## File Storage

- Files are stored using Laravel's file storage system
- File paths are returned as relative URLs
- Supported file types include images and documents
- Maximum file sizes:
  - General files: 5MB
  - Visitor photos: 2MB

## Security Considerations

1. **Authentication:** All protected endpoints require valid Bearer token
2. **Authorization:** Role-based access control implemented
3. **Data Filtering:** Users can only access their own data where applicable
4. **File Upload Security:** File type and size validation
5. **Input Validation:** Comprehensive validation rules for all inputs
6. **SQL Injection Protection:** Laravel ORM provides protection
7. **Cross-Origin Resource Sharing (CORS):** Configure as needed

## Development Notes

- Built with Laravel 11.31
- Uses Laravel Sanctum for API authentication
- Supports file uploads via multipart/form-data
- Implements comprehensive filtering and relationship loading
- Uses enum validation for status fields
- Automatic code generation for vehicles and locations
- Email notifications for password reset and visitor approvals

## Support

For technical support or questions about this API, please contact the development team or refer to the application's documentation.