<?php

namespace Database\Factories;

use App\Enums\VehicleType;
use App\Models\Flat;
use App\Models\Owner;
use App\Models\Vehicle;
use Illuminate\Database\Eloquent\Factories\Factory;

class VehicleFactory extends Factory
{
    protected $model = Vehicle::class;

    public function definition(): array
    {
        $vehicleType = $this->faker->randomElement(VehicleType::cases());
        
        return [
            'owner_id' => Owner::factory(),
            'flat_id' => Flat::factory(),
            'vehicle_type' => $vehicleType,
            'make' => $this->getVehicleMake($vehicleType),
            'model' => $this->getVehicleModel($vehicleType),
            'color' => $this->faker->randomElement(['White', 'Black', 'Silver', 'Red', 'Blue', 'Grey', 'Brown']),
            'registration_no' => $this->generateRegistrationNumber(),
            'unique_code' => $this->faker->unique()->regexify('[A-Z0-9]{8}'),
            'is_active' => $this->faker->boolean(90), // 90% chance of being active
        ];
    }

    private function getVehicleMake(VehicleType $type): string
    {
        return match ($type) {
            VehicleType::TWO_WHEELER => $this->faker->randomElement(['Honda', 'Bajaj', 'TVS', 'Yamaha', 'Hero', 'Royal Enfield']),
            VehicleType::FOUR_WHEELER => $this->faker->randomElement(['Maruti', 'Hyundai', 'Tata', 'Honda', 'Toyota', 'Mahindra', 'Ford']),
            VehicleType::OTHER => $this->faker->randomElement(['Mahindra', 'Tata', 'Ashok Leyland', 'Force']),
        };
    }

    private function getVehicleModel(VehicleType $type): string
    {
        return match ($type) {
            VehicleType::TWO_WHEELER => $this->faker->randomElement(['Activa', 'Splendor', 'Apache', 'FZ', 'Classic 350']),
            VehicleType::FOUR_WHEELER => $this->faker->randomElement(['Swift', 'i20', 'Nexon', 'City', 'Innova', 'XUV500']),
            VehicleType::OTHER => $this->faker->randomElement(['Bolero', 'Sumo', 'Dost', 'Tempo']),
        };
    }

    private function generateRegistrationNumber(): string
    {
        $states = ['MH', 'KA', 'TN', 'DL', 'UP', 'GJ', 'RJ', 'WB'];
        $state = $this->faker->randomElement($states);
        $district = $this->faker->numberBetween(1, 99);
        $series = $this->faker->randomLetter() . $this->faker->randomLetter();
        $number = $this->faker->numberBetween(1000, 9999);
        
        return sprintf('%s%02d%s%04d', $state, $district, $series, $number);
    }

    public function twoWheeler(): static
    {
        return $this->state(fn (array $attributes) => [
            'vehicle_type' => VehicleType::TWO_WHEELER,
            'make' => $this->getVehicleMake(VehicleType::TWO_WHEELER),
            'model' => $this->getVehicleModel(VehicleType::TWO_WHEELER),
        ]);
    }

    public function fourWheeler(): static
    {
        return $this->state(fn (array $attributes) => [
            'vehicle_type' => VehicleType::FOUR_WHEELER,
            'make' => $this->getVehicleMake(VehicleType::FOUR_WHEELER),
            'model' => $this->getVehicleModel(VehicleType::FOUR_WHEELER),
        ]);
    }

    public function active(): static
    {
        return $this->state(fn (array $attributes) => [
            'is_active' => true,
        ]);
    }

    public function inactive(): static
    {
        return $this->state(fn (array $attributes) => [
            'is_active' => false,
        ]);
    }
}